//
//  BMKBusRouteSearchResult.h
//  BaiduMapAPI_Search
//
//  Created by 王大川 on 2025/1/14.
//  Copyright © 2025 Baidu. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "BMKRouteSearchType.h"

@class BMKBusRouteStationInfo, BMKBusRouteNode, BMKBusRouteLine, BMKBusRouteStep, BMKBusTransferStationInfo, BMKBusTransferVehicleInfo, BMKMassTransferVehicleStation;

/// 公交路线方案类型
typedef enum {
    BMK_BUS_ROUTE_PLAN_TYPE_TRAIN               = 0, /// 火车
    BMK_BUS_ROUTE_PLAN_TYPE_TRAIN_HIGHSPEED     = 1, /// 高铁
    BMK_BUS_ROUTE_PLAN_TYPE_AIRPLANE            = 2, /// 飞机
    BMK_BUS_ROUTE_PLAN_TYPE_COACH               = 3, /// 大巴
    BMK_BUS_ROUTE_PLAN_TYPE_BUS                 = 4, /// 公交(包含区域一体化)
    BMK_BUS_ROUTE_PLAN_TYPE_WALK                = 5, /// 步行
    BMK_BUS_ROUTE_PLAN_TYPE_RIDE                = 6, /// 骑行
    BMK_BUS_ROUTE_PLAN_TYPE_TAXI                = 7, /// 打车
} BMKBusRoutePlanType;

/// 地铁线路开门方向类型
typedef enum {
    BMK_SUBWAY_SIDE_DOOR_TYPE_LEFT              = 1, /// 左侧
    BMK_SUBWAY_SIDE_DOOR_TYPE_RIGHT             = 2, /// 右侧
    BMK_SUBWAY_SIDE_DOOR_TYPE_ALL               = 3, /// 左右两侧
} BMKSubwaySideDoorType;

NS_ASSUME_NONNULL_BEGIN

/// 公交路线规划结果
@interface BMKBusRouteResult : NSObject

/// 路线类型，同城 or 跨城
@property (nonatomic, assign) BMKBusRouteType resultType;

/// 路线数量
@property (nonatomic, assign) NSInteger total;

/// 起点信息
@property (nonatomic, strong) BMKBusRouteNode *origin;

/// 终点信息
@property (nonatomic, strong) BMKBusRouteNode *destination;

/// 路线方案集合
@property (nonatomic, strong) NSArray<BMKBusRouteLine *> *routes;

/// 出租车信息，跨城不返回该字段
@property (nonatomic, strong) BMKTaxiInfo *taxiInfo;

/// 是否步行更优
@property (nonatomic, assign) BOOL isBetterForWalk;
@end

/// 公交路线规划的节点
@interface BMKBusRouteNode : BMKRouteNode

/// 城市名称，同城不返回该字段
@property (nonatomic, copy) NSString *cityName;

/// 城市id，同城不返回该字段
@property (nonatomic, copy) NSString *cityId;

/// 站点列表
@property (nonatomic, strong) NSArray<BMKBusRouteStationInfo *> *stationList;

@end


/// 公交路线规划站点信息
@interface BMKBusRouteStationInfo : BMKRouteNode

/// 大巴终点是否模糊
/// NO精确, YES模糊
@property (nonatomic, assign) BOOL isFuzzy;
/// 站点类型
@property (nonatomic, assign) BMKMassTransitType stationType;

@end

@interface BMKBusRouteLine : NSObject

/// 预计到达时间
/// 格式：yyyy-mm-dd hh:ss
@property (nonatomic, copy) NSString *arriveTime;
/// 出发时间
/// 格式：yyyy-mm-dd hh:ss
@property (nonatomic, copy) NSString *departTime;
/// 距离 单位：米
/// 跨城方案中， 不包含飞机、货车、大巴的距离
@property (nonatomic, assign) NSInteger distance;
/// 耗时 单位：秒
@property (nonatomic, assign) NSInteger duration;

/// 方案类型
@property (nonatomic, assign) BMKBusRoutePlanType planType;
/// 方案状态
/// 0 正常
/// 1 当前线路不确定是否运营
@property (nonatomic, assign) int planStatus;
/// 拥堵距离，单位：米，仅适用于城市内
@property (nonatomic, assign) NSInteger congestionDistance;
/// 拥堵耗时，单位：秒，仅适用于城市内
@property (nonatomic, assign) NSInteger congestionDuration;
/// 拥堵提示，仅适用于城市内
@property (nonatomic, copy) NSString *congestionText;
/// 最低价格，单位：分，适用于所有情况
@property (nonatomic, assign) int price;
/// 方案中的具体step
/// 二维数组，一个step内可能会出现多个选择，例如坐公交或打车
@property (nonatomic, strong) NSArray<NSArray<BMKBusRouteStep *> *> *steps;

@end

@interface BMKBusRouteStep : NSObject

/// 路段类型
/// 此类型不包含BMK_TRANSIT_SUBWAY
/// BMK_TRANSIT_BUSLINE 为一体化公交包含地铁
@property (nonatomic, assign) BMKMassTransitType stepType;
/// 当前step的指引文字描述，驾车无该字段
@property (nonatomic, copy) NSString *instruction;
/// 距离
/// 跨城方案中， 不包含飞机、货车、大巴的距离，该字段不会被返回
@property (nonatomic, assign) NSInteger distance;
/// 耗时，单位：秒
@property (nonatomic, assign) NSInteger duration;
/// 起点名
/// 步行
@property (nonatomic, copy) NSString *startName;
/// 起点地址
@property (nonatomic, copy) NSString *startAddress;
/// 起点坐标
/// 格式为[经度,纬度]
@property (nonatomic, assign) CLLocationCoordinate2D startLocation;
/// 终点名
@property (nonatomic, copy) NSString *endName;
/// 终点地址
/// 步行
@property (nonatomic, copy) NSString *endAddress;
/// 终点坐标
/// 格式为[经度,纬度]
@property (nonatomic, assign) CLLocationCoordinate2D endLocation;
/// 路段所经过的地理坐标集合
@property (nonatomic) BMKMapPoint *points;
/// 路段所经过的地理坐标集合内点的个数
@property (nonatomic, assign) int pointsCount;
/// 有票的最低票价
/// 跨城的step才返回，单位：分
@property (nonatomic, assign) NSInteger price;
/// 是否可以骑行
@property (nonatomic, assign) BOOL canRide;
/// 步行虚线类型
@property (nonatomic, assign) int walkType;
/// 换乘类型
@property (nonatomic, assign) BMKMassWalkType transType;
/// 换乘站点信息
@property (nonatomic, strong) NSArray<BMKBusTransferStationInfo *> *station;
/// 公交路线信息
@property (nonatomic, strong) BMKBusTransferVehicleInfo *vehicle;
/// 方案中的step
/// 在跨城结果中起点城市和终点城市的step中会返回
@property (nonatomic, strong) NSArray<NSArray<BMKBusRouteStep *> *> *lowerSteps;

@end

/// 公交路线中换乘站点信息
@interface BMKBusTransferStationInfo : NSObject
/// 站点耗时，单位：秒
@property (nonatomic, assign) NSInteger duration;
/// 站点UID
@property (nonatomic, copy) NSString *uid;
/// 站点名称
@property (nonatomic, copy) NSString *name;
/// 是否地下站点，0:否，1:是
@property (nonatomic, assign) BOOL underground;
/// 站点坐标
@property (nonatomic, assign) CLLocationCoordinate2D location;
@end

@interface BMKBusTransferVehicleInfo : NSObject
/// 线路名称
@property (nonatomic, copy) NSString *name;
/// 线路别名，跨城没有，一般是地铁才有别名
@property (nonatomic, copy) NSString *aliasName;
/// 同城交通工具类型，跨城时此值无效
@property (nonatomic, assign) BMKTransitType transitType;
/// 跨城交通工具类型，同城时此值无效
@property (nonatomic, assign) BMKMassTransitType massTransitType;
/// 线路UID
@property (nonatomic, copy) NSString *uid;
/// 出发站点UID
@property (nonatomic, copy) NSString *startUid;
/// 结束站点UID
@property (nonatomic, copy) NSString *endUid;
/// 线路方向，通常是线路的终点站
@property (nonatomic, copy) NSString *directText;
/// 地铁站入口，仅适用于城市内
/// 跨城的step无该字段，一般地铁线路才有该字段
@property (nonatomic, strong) BMKRouteNode *entrancePort;
/// 地铁站出口，仅适用于城市内
@property (nonatomic, strong) BMKRouteNode *exitPort;
/// 便捷换乘/出站信息，仅适用于城市内
/// 跨城的step无该字段，一般地铁线路才有该字段
@property (nonatomic, strong) NSArray<NSString *> *convientDoorInfo;
/// 开门方向文案，仅适用于城市内
@property (nonatomic, copy) NSString *sideDoorDesc;
/// 开门方向类型，仅适用于城市内
@property (nonatomic, assign) BMKSubwaySideDoorType sideDoorType;
/// 起点站点名称，仅适用于城市内
@property (nonatomic, copy) NSString *startName;
/// 结束站点名称，仅适用于城市内
@property (nonatomic, copy) NSString *endName;
/// 开始时间，格式依据同城或跨城不同而不同
/// 同城：表示早班车时间格式(HH:MM)
/// 跨城：表示到达时间
/// 格式(YYYY-MM-DD HH:MM)
@property (nonatomic, copy) NSString *startTime;
/// 结束时间，格式依据同城或跨城不同而不同
/// 同城：表示早班车时间格式(HH:MM)
/// 跨城：表示到达时间
/// 格式(YYYY-MM-DD HH:MM)
@property (nonatomic, copy) NSString *endTime;
/// 途经站点个数，仅适用于城市内
@property (nonatomic, assign) int stopNum;
/// 起点城市名称，仅适用于跨城
@property (nonatomic, copy) NSString *startCityName;
/// 终点城市名称，仅适用于跨城
@property (nonatomic, copy) NSString *endCityName;
/// 线路站点信息，仅适用于跨城
@property (nonatomic, strong) NSArray<BMKMassTransferVehicleStation *> *lineStations;
@end

@interface BMKMassTransferVehicleStation : BMKPlanNode

/// 发车时间，格式为HH:MM
@property (nonatomic, copy) NSString *startTime;
@end

NS_ASSUME_NONNULL_END
