/**
 * PaddleLite ImageProcess Wrapper
 */
#ifndef paddle_lite_image_preprocess_h
#define paddle_lite_image_preprocess_h

#include <cstdint>
#include <cstdio>
#include <vector>
#include "mml_inference_api.h"

namespace mml_framework {
namespace lite {

enum DataLayoutType {
  MML_kUnk = 0,
  MML_kNCHW = 1,
  MML_kNHWC = 3,
  MML_kImageDefault = 4,  // for opencl image2d
  MML_kImageFolder = 5,   // for opencl image2d
  MML_kImageNW = 6,       // for opencl image2d
  MML_kAny = 2,           // any data layout
  MML_NUM = 7,            // number of fields.
};

// color enum
enum ImageFormat {
  MML_RGBA = 0,
  MML_BGRA,
  MML_RGB,
  MML_BGR,
  MML_GRAY,
  MML_NV21 = 11,
  MML_NV12,
  MML_YUV420SP,
  MML_YUV420P,
  MML_YUV422,
  MML_YUV444
};
// flip enum
enum FlipParam {
  MML_XY = -1,  // flip along the XY axis
  MML_X = 0,    // flip along the X axis
  MML_Y         // flip along the Y axis
};
// transform param
typedef struct MML_NATIVE_API {
  int ih;                // input height
  int iw;                // input width
  int oh;                // outpu theight
  int ow;                // output width
  FlipParam flip_param;  // flip, support x, y, xy
  float rotate_param;    // rotate, support 90, 180, 270
} TransParam;

typedef mml_framework::MMLTensor Tensor;

class MML_NATIVE_API PaddleLiteImagePreprocess {
public:
  /*
   * init
   * param srcFormat: input image color
   * param dstFormat: output image color
   * param param: input image parameter, egs: input size
   */
  PaddleLiteImagePreprocess(ImageFormat srcFormat, ImageFormat dstFormat,
                            TransParam param);

  /*
   * image color convert
   * support NV12/NV21_to_BGR(RGB), NV12/NV21_to_BGRA(RGBA),
   * BGR(RGB)and BGRA(RGBA) transform,
   * BGR(RGB)and RGB(BGR) transform,
   * BGR(RGB)and RGBA(BGRA) transform,
   * BGR(RGB) and GRAY transform,
   * BGRA(RGBA) and GRAY transform,
   * param src: input image data
   * param dst: output image data
   */
  void imageConvert(const uint8_t *src, uint8_t *dst);

  /*
   * image color convert
   * support NV12/NV21_to_BGR(RGB), NV12/NV21_to_BGRA(RGBA),
   * BGR(RGB)and BGRA(RGBA) transform,
   * BGR(RGB)and RGB(BGR) transform,
   * BGR(RGB)and RGBA(BGRA) transform,
   * BGR(RGB)and GRAY transform,
   * BGRA(RGBA) and GRAY transform,
   * param src: input image data
   * param dst: output image data
   * param srcFormat: input image image format support: GRAY, NV12(NV21),
   * BGR(RGB) and BGRA(RGBA)
   * param dstFormat: output image image format, support GRAY, BGR(RGB) and
   * BGRA(RGBA)
   */
  void imageConvert(const uint8_t *src, uint8_t *dst, ImageFormat srcFormat,
                    ImageFormat dstFormat);

  /*
  * image color convert
  * support NV12/NV21_to_BGR(RGB), NV12/NV21_to_BGRA(RGBA),
  * BGR(RGB)and BGRA(RGBA) transform,
  * BGR(RGB)and RGB(BGR) transform,
  * BGR(RGB)and RGBA(BGRA) transform,
  * BGR(RGB)and GRAY transform,
  * BGRA(RGBA) and GRAY transform,
  * param src: input image data
  * param dst: output image data
  * param srcFormat: input image image format support: GRAY, NV12(NV21),
  * BGR(RGB) and BGRA(RGBA)
  * param dstFormat: output image image format, support GRAY, BGR(RGB) and
  * BGRA(RGBA)
  * param srcw: input image width
  * param srch: input image height
  */
  void imageConvert(const uint8_t *src,
                    uint8_t *dst,
                    ImageFormat srcFormat,
                    ImageFormat dstFormat,
                    int srcw,
                    int srch);

  /*
   * image resize, use bilinear method
   * support image format: 1-channel image (egs: GRAY, 2-channel image (egs:
   * NV12, NV21), 3-channel(egs: BGR), 4-channel(egs: BGRA)
   * param src: input image data
   * param dst: output image data
   */
  void imageResize(const uint8_t *src, uint8_t *dst);

  /*
   image resize, use bilinear method
  * support image format: 1-channel image (egs: GRAY, 2-channel image (egs:
  NV12, NV21), 3-channel image(egs: BGR), 4-channel image(egs: BGRA)
  * param src: input image data
  * param dst: output image data
  * param srcw: input image width
  * param srch: input image height
  * param dstw: output image width
  * param dsth: output image height
  */
  void imageResize(const uint8_t *src, uint8_t *dst, ImageFormat srcFormat,
                   int srcw, int srch, int dstw, int dsth);

  /*
   * image Rotate
   * support 90, 180 and 270 Rotate process
   * color format support 1-channel image, 3-channel image and 4-channel image
   * param src: input image data
   * param dst: output image data
   */
  void imageRotate(const uint8_t *src, uint8_t *dst);

  /*
   * image Rotate
   * support 90, 180 and 270 Rotate process
   * color format support 1-channel image, 3-channel image and 4-channel image
   * param src: input image data
   * param dst: output image data
   * param srcFormat: input image format, support GRAY, BGR(RGB) and BGRA(RGBA)
   * param srcw: input image width
   * param srch: input image height
   * param degree: Rotate degree, support 90, 180 and 270
   */
  void imageRotate(const uint8_t *src, uint8_t *dst, ImageFormat srcFormat,
                   int srcw, int srch, float degree);

  /*
   * image Flip
   * support X, Y and XY flip process
   * color format support 1-channel image, 3-channel image and 4-channel image
   * param src: input image data
   * param dst: output image data
   */
  void imageFlip(const uint8_t *src, uint8_t *dst);

  /*
  * image crop process
  * color format support 1-channel image, 3-channel image and 4-channel image
  * param src: input image data
  * param dst: output image data
  */
  void imageCrop(const uint8_t *src,
                 uint8_t *dst,
                 ImageFormat srcFormat,
                 int srcw,
                 int srch,
                 int left_x,
                 int left_y,
                 int dstw,
                 int dsth);

  /*
   * image Flip
   * support X, Y and XY flip process
   * color format support 1-channel image, 3-channel image and 4-channel image
   * param src: input image data
   * param dst: output image data
   * param srcFormat: input image format, support GRAY, BGR(RGB) and BGRA(RGBA)
   * param srcw: input image width
   * param srch: input image height
   * param flip_param: flip parameter, support X, Y and XY
   */
  void imageFlip(const uint8_t *src, uint8_t *dst, ImageFormat srcFormat,
                 int srcw, int srch, FlipParam flip_param);

  /*
   * change image data to tensor data
   * support image format is GRAY, BGR(RGB) and BGRA(RGBA), Data layout is NHWC
   * and
   * NCHW
   * param src: input image data
   * param dstTensor: output tensor data
   * param layout: output tensor layout，support NHWC and NCHW
   * param means: means of image
   * param scales: scales of image
   */
  void image2Tensor(const uint8_t *src, Tensor *dstTensor,
                    DataLayoutType layout, float *means, float *scales);

  /*
   * change image data to tensor data
   * support image format is GRAY, BGR(RGB) and BGRA(RGBA), Data layout is NHWC
   * and
   * NCHW
   * param src: input image data
   * param dstTensor: output tensor data
   * param srcFormat: input image format, support BGR(RGB) and BGRA(RGBA)
   * param srcw: input image width
   * param srch: input image height
   * param layout: output tensor layout，support NHWC and NCHW
   * param means: means of image
   * param scales: scales of image
   */
  void image2Tensor(const uint8_t *src, Tensor *dstTensor,
                    ImageFormat srcFormat, int srcw, int srch,
                    DataLayoutType layout, float *means, float *scales);

  /**
   * 析构函数
   */
  ~PaddleLiteImagePreprocess();

private:
  void *preprocesser_;
};

}  // namespace lite
}  // namespace mml_framework
#endif  /* paddle_lite_image_preprocess_h */
