//
//  BMKLocation.h
//  BMKLocationSDK
//
//  Created by baidu on 2018/1/29.
//  Copyright © 2018年 baidu. All rights reserved.
//

#ifndef BMKLocation_h
#define BMKLocation_h

#import <CoreLocation/CoreLocation.h>
#import "BMKLocationReGeocode.h"

/**
 * BMKLocationProvider 位置数据来源，分iOS系统定位和其他定位服务结果两种，目前仅支持iOS系统定位服务
 *
 */
typedef NS_ENUM(int, BMKLocationProvider) {
    
    BMKLocationProviderIOS = 0,           //!<位置来源于iOS本身定位
    BMKLocationProviderOther          //!<位置来源于其他定位服务
    
};

/**
 * BMKMockLocationProbability 位置数据为作弊定位的概率
 *
 */
typedef NS_ENUM(int, BMKMockLocationProbability) {
    
    BMKMockLocationProbabilityNone = 0,           //!<作弊概率为0
    BMKMockLocationProbabilityLow,        //!<低概率
    BMKMockLocationProbabilityMid,        //!<中概率
    BMKMockLocationProbabilityHigh        //!<高概率
};

/**
 * BMKLocationIndoorParkState 室内定位状态
 *
 */
typedef NS_ENUM(NSInteger, BMKLocationIndoorParkState)
{
    BMKLocationIndoorParkStateUnknown = -1,        ///<地库状态未知
    BMKLocationIndoorParkStateFalse = 0,           ///<不是地库状态
    BMKLocationIndoorParkStateTrue  = 1,           ///<是地库状态
    BMKLocationIndoorParkStateInUndergroundPipeline  = 2           ///<是地下环廊
};


@interface BMKLocation : CLLocation


///BMKLocation 地址数据
@property(nonatomic, copy) BMKLocationReGeocode * _Nullable rgcData;

///BMKLocation 位置来源
@property(nonatomic, assign) BMKLocationProvider provider;

///BMKLocation 位置ID
@property(nonatomic, retain) NSString * _Nullable locationID;

///BMKLocation 位置作弊概率
@property(nonatomic, assign) BMKMockLocationProbability mockProbability;

/*
 *  floorString
 *
 *  Discussion:
 *    室内定位成功时返回的楼层信息，ex:f1
 */
@property(nonatomic, copy, nullable) NSString *floorString;

/*
 *  floorValue
 *
 *  Discussion:
 *    这个是苹果系统返回的楼层数据，不受百度控制，便于外面开发者使用
 */
@property(nonatomic, copy, nullable) CLFloor *floorValue;

/*
 *  buildingID
 *
 *  Discussion:
 *    室内定位成功时返回的百度建筑物ID
 */
@property(nonatomic, copy, nullable) NSString *buildingID;

/*
 *  buildingName
 *
 *  Discussion:
 *    室内定位成功时返回的百度建筑物名称
 */
@property(nonatomic, copy, nullable) NSString *buildingName;

/*
 *  in_out_ble_flag
 *
 *  Discussion:
 *    高精蓝牙定位室内外判断 0 是室外 1 室内   -1 是未知
 */
@property(nonatomic, assign) int bleInOutFlag;


/*
 *  extraInfo
 *
 *  Discussion:
 *    定位附加信息，如停车位code识别结果、停车位code示例、vdr推算结果置信度等
 */
@property(nonatomic, copy, nullable) NSDictionary * extraInfo;

/*
 *  signalType
 *
 *  Discussion:
 *    信号类型 取的是CLLocation私有属性type的值
 * 0  unknown 应用程序生成的定位数据，一般在越狱设备下，通过虚拟定位程序来生成。
 * 1  gps  GPS生成的定位数据
 * 2  nmea
 * 3  accessory 蓝牙等外部设备模拟定位生成的定位数据
 * 4  wifi  WIFI定位生成的数据
 * 5  skyhook WIFI定位生成的数据
 * 6  cell 手机基站定位生成的数据
 * 7  lac LAC生成的定位数据
 * 8  mcc
 * 9  gpscoarse
 * 10 pipeline
 * 11 max
 */
@property (nonatomic, assign) int signalType;

/*
 *  rawCoordinateGather
 *
 *  Discussion:
 *    原始坐标 取的是CLLocation私有属性rawCoordinate的值
 */
@property(nonatomic) CLLocationCoordinate2D rawCoordinateGather;


/*
 *  locationSignalEnvironmentType
 *
 *  Discussion:
 *    跟踪运行信息 取的是CLLocation私有属性signalEnvironmentType的值
 */
@property (nonatomic, assign) int locationSignalEnvironmentType;

/**
 *  @brief 初始化BMKLocation实例
 *  @param loc CLLocation对象
 *  @param rgc BMKLocationReGeocode对象
 *  @return BMKLocation id
 */
- (id _Nonnull)initWithLocation:(CLLocation * _Nullable)loc withRgcData:(BMKLocationReGeocode * _Nullable)rgc;

/**
 *  @brief 构造BMKLocation
 *  @param location CLLocation
 *  @param floorString 楼层字符串
 *  @param buildingID 建筑物ID
 *  @param buildingName 建筑物名称
 *  @param info 位置附加信息
 *  @return BMKLocation id
 */
-(id _Nonnull)initWithLocation:(CLLocation * _Nullable)location floorString:(NSString * _Nullable)floorString buildingID:(NSString * _Nullable)buildingID
                  buildingName:(NSString * _Nullable)buildingName extraInfo:(NSDictionary * _Nullable)info withRgcData:(BMKLocationReGeocode * _Nullable)rgc;

/**
 *  @brief 构造BMKLocation
 *  @param location CLLocation
 *  @param floorString 楼层字符串
 *  @param buildingID 建筑物ID
 *  @param buildingName 建筑物名称
 *  @param info 位置附加信息
 *  @return BMKLocation id
 */
-(id _Nonnull)initWithLocation:(CLLocation * _Nullable)location floorString:(NSString * _Nullable)floorString buildingID:(NSString * _Nullable)buildingID
         buildingName:(NSString * _Nullable)buildingName extraInfo:(NSDictionary * _Nullable)info;

/**
 *  @brief 构造BMKLocation
 *  @param location CLLocation
 *  @param floorString 楼层字符串
 *  @param buildingID 建筑物ID
 *  @param buildingName 建筑物名称
 *  @param info 位置附加信息
 *  @param pri_pro_switch vdr隐私属性解析开关
 *  @return BMKLocation id
 */
-(id _Nonnull)initWithLocation:(CLLocation * _Nullable)location floorString:(NSString * _Nullable)floorString buildingID:(NSString * _Nullable)buildingID
         buildingName:(NSString * _Nullable)buildingName extraInfo:(NSDictionary * _Nullable)info withPri_pro_switch:(BOOL)pri_pro_switch;


@end
#endif /* BMKLocation_h */
